function LWM_ini = LWM_RND_initialization(oprice,strike, rf, TTM,Ft,weights)
%==========================================================================================
%The function performs the initialization step of the Global search
%algorithm of Li, Nolte and Pham (2023). In particular, it computes 
% the 1-mixture Lognormal and Weibull fit and obtain the bounds for the objective
% function value.
%
%INPUT:
%   oprice: N-by-1 call option prices
%   strike: N-by-1 strike prices of the options
%   rf: risk-free rate
%   TTM: time to maturity of the options (in years)
%   Ft: underlying futures price
%   weights: N-by-1 weight vector. If not provided, assumed equal weights
%
%OUTPUT:
%   LWM_ini: a struct consisting of
%          kfit: fitted k value for 1-mixture Weibull distribution
%          sfit: fitted s value for 1-mixture Lognormal distribution
%          fvalk: minimized objective function value for the 1-mixture Weibull distribution
%          fvals: minimized objective function value for the 1-mixture Lognormal distribution
%          IV_atm: at the money Black implied volatility 
%          x0k: Empty array [] - estimated parameters for 1-mixture Weibull distribution.
%          x0s: Empty array [] - estimated parameters for 1-mixture Lognormal distribution.
%==========================================================================================
% This ver: 2023/05/24
% Authors: Yifan Li (yifan.li@manchester.ac.uk)
%          Ingmar Nolte (i.nolte@lancaster.ac.uk)
%          Manh Pham (m.c.pham@lancaster.ac.uk)
% Reference: Li, Y., Nolte, I., and Pham, M. C. (2023). Parametric Risk-Neutral 
%          Density Estimation via Finite Lognormal-Weibull Mixtures
%========================================================================================== 

if nargin < 6
    weights = ones(size(oprice));
end

W = diag(weights);
o_ATM = interp1(strike,oprice,Ft,'pchip'); %interpolating the ATM option price
IV_atm = sqrt(8/TTM)*erfinv(o_ATM/Ft);
%closed-form formula at the money implied volatility. One can also use:
% IV_atm = blkimpv(Ft,Ft,rf,TTM,o_ATM);

option_ini = optimoptions('fmincon','display','off');
f=@(x)  (WB_f(Ft, strike, rf,TTM, x,false,false)-oprice)'*W*...
    (WB_f(Ft, strike, rf,TTM, x,false,false)-oprice);
[kfit,fvalk] =fmincon(f,s2k(IV_atm,TTM),[],[],[],[],0,1,[],option_ini);
g=@(x)  (LN_f(Ft, strike, rf,TTM, x,false,false)-oprice)'*W* ...
    (LN_f(Ft, strike, rf,TTM, x,false,false)-oprice);
[sfit,fvals] =fmincon(g,IV_atm,[],[],[],[],0,1,[],option_ini);

LWM_ini.kfit = kfit;
LWM_ini.sfit = sfit;
LWM_ini.fvalk = fvalk;
LWM_ini.fvals = fvals;
LWM_ini.IV_atm = IV_atm;
LWM_ini.x0s = [];
LWM_ini.x0k = [];

end

